import os
from PIL import Image

# Ensure Pillow supports DDS
try:
    Image.register_extensions(Image.EXTENSION, {'.dds': 'DDS'})
except AttributeError:
    print("Pillow might not support DDS out of the box. Consider installing 'pillow-dds'.")

# Function to check if an alpha channel is entirely white
def is_alpha_channel_white(img):
    if img.mode == "RGBA":
        # Split the image into bands
        _, _, _, alpha = img.split()
        # Check if all pixels in the alpha channel are white
        return all(pixel == 255 for pixel in alpha.getdata())
    return False

# Function to convert DDS to TGA
def convert_dds_to_tga(folder):
    # Iterate through files in the folder
    for file_name in os.listdir(folder):
        if file_name.lower().endswith('.dds'):
            file_path = os.path.join(folder, file_name)
            output_file = os.path.splitext(file_name)[0] + '.tga'
            output_path = os.path.join(folder, output_file)
            
            try:
                # Open the DDS file
                with Image.open(file_path) as img:
                    # Check for alpha channel and determine if it's entirely white
                    if img.mode in ("RGBA", "LA") or ('A' in img.getbands()):
                        if img.mode != "RGBA":
                            img = img.convert("RGBA")
                        if is_alpha_channel_white(img):
                            img = img.convert("RGB")  # Remove alpha channel if entirely white
                    else:
                        img = img.convert("RGB")

                    # Save as TGA
                    img.save(output_path, format="TGA")
                    print(f"Converted: {file_name} -> {output_file}")
            except Exception as e:
                print(f"Failed to convert {file_name}: {e}")

if __name__ == "__main__":
    # Get the current folder of the script
    folder = os.path.dirname(os.path.abspath(__file__))
    convert_dds_to_tga(folder)
